<?php

namespace App\Model\Table;

class RegisterTable
{
    public function initialize(array $config): void
    {
        parent::initialize($config);
        $this->setTable('logins');
        $this->setPrimaryKey('id');
        $this->addBehavior('Timestamp');
    }

    public function validationDefault(Validator $v): Validator
    {
        $v->scalar('first_name')->maxLength('first_name', 50)->requirePresence('first_name', 'create')->notEmptyString('first_name');
        $v->scalar('last_name')->maxLength('last_name', 50)->allowEmptyString('last_name');

        $v->email('email')->requirePresence('email', 'create')->notEmptyString('email');

        $v->scalar('mobile_phone')
            ->requirePresence('mobile_phone', 'create')->notEmptyString('mobile_phone', 'Mobile phone is required')
            ->add('mobile_phone', 'format', [
                'rule' => function ($value) {
                    return (bool)preg_match('/^\+?\d[\d\s-]{7,15}$/', (string)$value);
                },
                'message' => 'Enter a valid phone number (e.g. +61 xxx xxx xxx).'
            ])->maxLength('mobile_phone', 32);

        $v->date('date_of_birth')->requirePresence('date_of_birth', 'create')->normalizeDob('date_of_birth', 'Date of birth is required');

        $v->scalar('password')
            ->minLength('password', 8, 'Password must be at least 8 characters.')
            ->requirePresence('password', 'create')->notEmptyString('password')
            ->add('password', 'strength', [
                'rule' => function ($value) {
                    $v = (string)$value;
                    return (bool)(preg_match('/[A-Z]/', $v) && preg_match('/\d/', $v) && strlen($v) >= 8);
                },
                'message' => 'Password must contain at least 1 uppercase letter and 1 number.'
            ]);

        $v->boolean('terms_agreed')->requirePresence('terms_agreed', 'create')
            ->add('terms_agreed', 'accepted', ['rule' => fn($x) => (int)$x === 1, 'message' => 'You must agree to the terms.']);

        return $v;
    }


    public function buildRules(RulesChecker $rules): RulesChecker
    {
        $rules->add($rules->isUnique(['email'], 'This email is already used.'));
        return $rules;
    }
}
