<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Table;
use Cake\Validation\Validator;
use Cake\ORM\RulesChecker;

class LoginTable extends Table
{
    public function initialize(array $config): void
    {
        parent::initialize($config);
        $this->setTable('logins');
        $this->setPrimaryKey('id');
        $this->addBehavior('Timestamp');

        $this->belongsToMany('Participants', [
            'joinTable'        => 'logins_participants',
            'foreignKey'       => 'login_id',
            'targetForeignKey' => 'participant_id',
        ]);

        $this->belongsToMany('Specializations', [
            'foreignKey'        => 'login_id',
            'targetForeignKey'  => 'specialization_id',
            'joinTable'         => 'clinicians_specializations',
        ]);

        $this->hasMany('ClinicUsers', ['foreignKey' => 'login_id']);
    }

    public function validationDefault(Validator $v): Validator
    {
        $v->scalar('first_name')->maxLength('first_name', 50)->requirePresence('first_name', 'create')->notEmptyString('first_name');
        $v->scalar('last_name')->maxLength('last_name', 50)->allowEmptyString('last_name');

        $v->email('email')->requirePresence('email', 'create')->notEmptyString('email');

        $v->scalar('mobile_phone')
            ->requirePresence('mobile_phone', 'create')->notEmptyString('mobile_phone', 'Mobile phone is required')
            ->add('mobile_phone', 'format', [
                'rule' => fn($value) => (bool)preg_match('/^\+?\d[\d\s-]{7,15}$/', (string)$value),
                'message' => 'Enter a valid phone number (e.g. +61 xxx xxx xxx).'
            ])->maxLength('mobile_phone', 32);

        $v->date('date_of_birth')->requirePresence('date_of_birth', 'create')->notEmptyDate('date_of_birth', 'Date of birth is required');

        $v->scalar('password')
            ->minLength('password', 8, 'Password must be at least 8 characters.')
            ->requirePresence('password', 'create')->notEmptyString('password')
            ->add('password', 'strength', [
                'rule' => fn($value) => (bool)(preg_match('/[A-Z]/', (string)$value) && preg_match('/\d/', (string)$value) && strlen((string)$value) >= 8),
                'message' => 'Password must contain at least 1 uppercase letter and 1 number.'
            ]);

        $v->boolean('terms_agreed')->requirePresence('terms_agreed', 'create')
            ->add('terms_agreed', 'accepted', ['rule' => fn($x) => (int)$x === 1, 'message' => 'You must agree to the terms.']);

        $v->scalar('role')->maxLength('role', 20)
            ->allowEmptyString('role')
            ->inList('role', ['admin','clinician'], 'Invalid role');

        $v->scalar('emergency_name')
            ->maxLength('emergency_name', 120)
            ->allowEmptyString('emergency_name');

        $v->email('emergency_email')
            ->allowEmptyString('emergency_email');

        $v->scalar('emergency_phone')
            ->maxLength('emergency_phone', 40)
            ->allowEmptyString('emergency_phone')
            ->add('emergency_phone', 'format', [
                'rule'    => function ($value) {
                    if ($value === null || $value === '') return true;
                    return (bool)preg_match('/^\+?\d[\d\s\-]{7,20}$/', (string)$value);
                },
                'message' => 'Please enter a valid emergency phone number.'
            ]);

        return $v;
    }

    public function buildRules(RulesChecker $rules): RulesChecker
    {
        $rules->add($rules->isUnique(['email'], 'This email is already used.'));
        return $rules;
    }
}
