<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\Http\Response;
use Cake\ORM\TableRegistry;

class OnboardingController extends AppController
{
    public function initialize(): void
    {
        parent::initialize();
        $this->Clinics          = $this->fetchTable('Clinics');
        $this->ClinicLocations  = $this->fetchTable('ClinicLocations');
        $this->ClinicPhones     = $this->fetchTable('ClinicPhones');
        $this->ClinicServices = $this->fetchTable('ClinicServices');
        $this->OnboardingStates = $this->fetchTable('OnboardingStates');
        $this->viewBuilder()->setLayout('default');
    }

    /** Resolve clinic id from session or the first clinic for the current user. */
    private function resolveClinicId(): ?int
    {
        $sess = $this->request->getSession();
        $cid  = $sess->read('Current.clinic_id');
        if ($cid) return (int)$cid;

        $identity = $this->request->getAttribute('identity');
        if ($identity) {
            $loginId = (int)$identity->getIdentifier();
            $ClinicUsers = $this->fetchTable('ClinicUsers'); // maps to clinic_users
            $row = $ClinicUsers->find()
                ->select(['clinic_id'])
                ->where(['login_id' => $loginId])
                ->orderDesc('role')
                ->first();

            if ($row && $row->clinic_id) {
                $cid = (int)$row->clinic_id;
                $sess->write('Current.clinic_id', $cid);
                return $cid;
            }
        }
        return null;
    }

    /** Get or create the onboarding state row for the clinic. */
    private function getOrCreateState(int $clinicId)
    {
        $state = $this->OnboardingStates->find()
            ->where(['clinic_id' => $clinicId])
            ->first();

        if (!$state) {
            $state = $this->OnboardingStates->newEntity([
                'clinic_id'  => $clinicId,
                'dismissed'  => 0,
                'step1_done' => 0,
                'step2_done' => 0,
                'step3_done' => 0,
                'step4_done' => 0,
            ]);
            $this->OnboardingStates->save($state);
        }
        return $state;
    }

    public function index()
    {
        return $this->redirect(['controller' => 'Calendar', 'action' => 'index']);
    }

    /** NEW: AJAX body for the modal */
    public function modal()
    {
        $this->request->allowMethod(['get']);

        $cid = $this->resolveClinicId();
        if (!$cid) {
            return $this->response->withStatus(400)->withStringBody('No clinic selected');
        }

        // --- Clinics row (but don't assume address_* exists) ---
        $clinic = null;
        $clinicCols = [];
        try {
            $clinicCols = $this->Clinics->getSchema()->columns();
        } catch (\Throwable $e) {
            $clinicCols = [];
        }

        $select = ['id'];
        $hasBusinessPhone = in_array('business_phone', $clinicCols, true);
        if ($hasBusinessPhone) {
            $select[] = 'business_phone';
        }

        $select = ['id', 'name'];
        if ($hasBusinessPhone) {
            $select[] = 'business_phone';
        }

        $clinic = $this->Clinics->find()
            ->select($select)
            ->where(['id' => $cid])
            ->first();

        // --- Locations: primary first (is_primary DESC, then id ASC) ---
        $locRows = $this->ClinicLocations->find()
            ->where(['clinic_id' => $cid])
            ->order(['is_primary' => 'DESC', 'id' => 'ASC'])
            ->all()
            ->toList();

        $primaryLoc = $locRows[0] ?? null;
        $extraLocs  = array_slice($locRows, 1);

        // Fallback: if there are no rows in clinic_locations, show empty fields
        if (!$primaryLoc) {
            $primaryLoc = (object)[
                'street'   => '',
                'suburb'   => '',
                'state'    => '',
                'postcode' => '',
                'country'  => '',
            ];
        }

        // --- Phones: primary first (is_primary DESC, id ASC) ---
        $phoneRows = $this->ClinicPhones->find()
            ->where(['clinic_id' => $cid])
            ->order(['is_primary' => 'DESC', 'id' => 'ASC'])
            ->all()
            ->toList();

        $services = $this->ClinicServices->find()
            ->where(['clinic_id' => $cid])
            ->orderAsc('name')
            ->all()
            ->toList();

        $primaryPhone = $phoneRows[0]->phone ?? ($hasBusinessPhone ? (string)($clinic->business_phone ?? '') : '');
        $extraPhones  = array_map(fn($r) => $r->phone, array_slice($phoneRows, 1));

        $state = $this->getOrCreateState($cid);
        $this->set(compact('state','primaryLoc','extraLocs','primaryPhone','extraPhones','services','clinic'));
        $this->viewBuilder()->setLayout('ajax');
        return $this->render('modal');
    }

    /** Mark a step done (called from the modal). */
    public function completeStep(?int $step = null)
    {
        $this->request->allowMethod(['get','post','patch','put']);

        if (!$step || $step < 1 || $step > 2) {
            return $this->response->withStatus(400)->withType('application/json')
                ->withStringBody(json_encode(['ok' => false, 'error' => 'Invalid step']));
        }

        $cid = $this->resolveClinicId();
        if (!$cid) {
            return $this->response->withStatus(400)->withType('application/json')
                ->withStringBody(json_encode(['ok' => false, 'error' => 'No clinic selected']));
        }

        $state = $this->getOrCreateState($cid);
        $field = 'step' . $step . '_done';
        if (!(int)$state->get($field)) {
            $state->set($field, 1);
            $this->OnboardingStates->save($state);
        }

        $done    = (int)$state->step1_done + (int)$state->step2_done;
        $percent = (int)round($done * 50);

        return $this->response->withType('application/json')
            ->withStringBody(json_encode(['ok' => true, 'step' => $step, 'percent' => $percent]));
    }

    public function dismiss()
    {
        $cid = $this->resolveClinicId();
        if (!$cid) return $this->redirect(['controller' => 'Calendar', 'action' => 'index']);

        $state = $this->getOrCreateState($cid);
        $state->dismissed = 1;
        $this->OnboardingStates->save($state);

        return $this->redirect(['controller' => 'Calendar', 'action' => 'index']);
    }

    public function bizInfo()
    {
        $this->request->allowMethod(['get']);
        $cid = $this->resolveClinicId();
        if (!$cid) return $this->response->withStatus(400)->withStringBody('No clinic selected');

        $Clinics = $this->fetchTable('Clinics');
        $clinic  = $Clinics->find()->where(['id' => (int)$cid])->first();

        $this->set(compact('clinic'));
        $this->viewBuilder()->setLayout('ajax');
        return $this->render('bizinfo');
    }

    public function savePractice(): Response
    {
        $this->request->allowMethod(['post']);

        $cid = $this->resolveClinicId();
        if (!$cid) {
            return $this->response->withStatus(400)->withType('json')
                ->withStringBody(json_encode(['ok' => false, 'error' => 'No clinic selected']));
        }

        $clinicName = trim((string)$this->request->getData('clinic_name', ''));

        if ($clinicName !== '') {
            $this->Clinics->updateAll(['name' => $clinicName], ['id' => $cid]);
        }

        $locations = (array)($this->request->getData('locations') ?? []);
        if (empty($locations) || !isset($locations[0])) {
            return $this->response->withStatus(422)->withType('json')
                ->withStringBody(json_encode(['ok' => false, 'error' => 'No location provided']));
        }

        ksort($locations, SORT_NUMERIC);
        $this->ClinicLocations->deleteAll(['clinic_id' => $cid]);

        $i = 0;
        foreach ($locations as $loc) {
            $row = $this->ClinicLocations->newEntity([
                'clinic_id'  => $cid,
                'label'      => null,
                'street'     => (string)($loc['address_street']   ?? ''),
                'suburb'     => (string)($loc['address_suburb']   ?? ''),
                'state'      => (string)($loc['address_state']    ?? ''),
                'postcode'   => (string)($loc['address_postcode'] ?? ''),
                'country'    => (string)($loc['address_country']  ?? ''),
                'is_primary' => $i === 0 ? 1 : 0,
            ]);
            $this->ClinicLocations->saveOrFail($row);
            $i++;
        }

        // ✅ Get latest name from DB (to confirm saved and return to frontend)
        $clinic = $this->Clinics->find()
            ->select(['name'])
            ->where(['id' => $cid])
            ->first();

        return $this->response->withType('json')
            ->withStringBody(json_encode([
                'ok' => true,
                'business_name' => $clinic?->name ?? $clinicName
            ]));
    }

    public function savePhone(): Response
    {
        $this->request->allowMethod(['post']);
        $cid = $this->resolveClinicId();
        if (!$cid) {
            return $this->response->withStatus(400)->withType('json')
                ->withStringBody(json_encode(['ok'=>false,'error'=>'No clinic selected']));
        }

        // Accept either name
        $raw = $this->request->getData('phones');
        if ($raw === null) {
            $raw = $this->request->getData('business_phone'); // because form uses business_phone[]
        }

        $phones = array_values(array_filter(
            array_map('trim', (array)$this->request->getData('phones')),
            fn($p) => $p !== ''
        ));

        if (empty($phones)) {
            return $this->response->withStatus(422)->withType('json')
                ->withStringBody(json_encode(['ok'=>false,'error'=>'No phone provided']));
        }

        $clinic = $this->Clinics->get($cid);
        $clinic->business_phone = $phones[0]; // primary
        $this->Clinics->saveOrFail($clinic);

        $this->ClinicPhones->deleteAll(['clinic_id' => $cid]);
        foreach ($phones as $i => $p) {
            $this->ClinicPhones->save(
                $this->ClinicPhones->newEntity([
                    'clinic_id'  => $cid,
                    'phone'      => $p,
                    'is_primary' => $i === 0 ? 1 : 0,
                ])
            );
        }

        return $this->response->withType('json')
            ->withStringBody(json_encode(['ok'=>true]));
    }

    public function reset()
    {
        $cid = $this->resolveClinicId();
        $OnboardingStates = $this->fetchTable('OnboardingStates');
        $OnboardingStates->updateAll(['dismissed' => 0], ['clinic_id' => $cid]);
        $this->Flash->success('Onboarding reset.');
        return $this->redirect($this->referer() ?: ['controller' => 'Dashboard', 'action' => 'index']);
    }

    public function servicesModal()
    {
        $this->request->allowMethod(['get']);

        $cid = $this->resolveClinicId();
        if (!$cid) {
            return $this->response->withStatus(400)->withStringBody('No clinic selected');
        }

        $ClinicServices = TableRegistry::getTableLocator()->get('ClinicServices');

        $services = $ClinicServices->find()
            ->where(['clinic_id' => $cid])
            ->orderAsc('name')
            ->all()
            ->toList();

        $this->set(compact('services'));
        $this->viewBuilder()->setLayout('ajax');
        return $this->render('services_modal'); // templates/Onboarding/services_modal.php
    }

    public function saveServices()
    {
        $this->request->allowMethod(['post']);
        $cid = $this->resolveClinicId();
        if (!$cid) {
            return $this->response->withStatus(400)->withType('json')
                ->withStringBody(json_encode(['ok' => false, 'error' => 'No clinic selected']));
        }

        $rows = (array)$this->request->getData('services');
        // Strategy: wipe and re-insert (simple, consistent), or upsert. Keeping it simple:
        $this->ClinicServices->deleteAll(['clinic_id' => $cid]);

        foreach ($rows as $r) {
            $r = array_map(static fn($v) => is_string($v) ? trim($v) : $v, $r);
            if (($r['name'] ?? '') === '') continue; // skip empties

            $entity = $this->ClinicServices->newEmptyEntity();
            $entity->clinic_id = $cid;
            $entity->name = (string)($r['name'] ?? '');
            $entity->code = (string)($r['code'] ?? '');
            $entity->duration_minutes = (int)($r['duration_minutes'] ?? 0);
            $entity->price_cents = ($r['price_cents'] === '' || $r['price_cents'] === null) ? 0 : (int)$r['price_cents'];
            $entity->active = 1;

            if (!$this->ClinicServices->save($entity)) {
                return $this->response->withStatus(422)->withType('json')
                    ->withStringBody(json_encode(['ok' => false, 'error' => 'Save failed']));
            }
        }

        return $this->response->withType('json')->withStringBody(json_encode(['ok' => true]));
    }

    public function services(): Response
    {
        $this->request->allowMethod(['get']);

        $cid = $this->resolveClinicId();
        if (!$cid) {
            return $this->response->withStatus(400)->withType('json')
                ->withStringBody(json_encode(['ok' => false, 'error' => 'No clinic selected']));
        }

        $rows = $this->ClinicServices->find()
            ->select(['id','name','code','duration_minutes','price_cents','active'])
            ->where(['clinic_id' => $cid, 'active' => 1])
            ->orderAsc('name')
            ->all();

        $out = [];
        foreach ($rows as $r) {
            $out[] = [
                'id'               => (int)$r->id,
                'name'             => (string)$r->name,
                'code'             => (string)($r->code ?? ''),
                'duration_minutes' => (int)($r->duration_minutes ?? 0),
                'price_cents'      => (int)($r->price_cents ?? 0),
            ];
        }

        return $this->response->withType('json')
            ->withStringBody(json_encode($out));
    }

    public function getPracticeInfo()
    {
        $this->request->allowMethod(['get']);

        // Get current clinic ID (using your existing helper)
        $cid = $this->resolveClinicId();

        if (!$cid) {
            return $this->response->withStatus(400)->withType('json')
                ->withStringBody(json_encode([
                    'ok' => false,
                    'error' => 'No clinic selected'
                ]));
        }

        // Fetch the clinic record from the database
        $clinic = $this->Clinics->find()
            ->select(['id', 'name'])
            ->where(['id' => $cid])
            ->first();

        if (!$clinic) {
            return $this->response->withStatus(404)->withType('json')
                ->withStringBody(json_encode([
                    'ok' => false,
                    'error' => 'Clinic not found'
                ]));
        }

        // Return JSON response with the business name
        $this->set([
            'ok' => true,
            'business_name' => $clinic->name,
            '_serialize' => ['ok', 'business_name']
        ]);
    }
}
