<?php
declare(strict_types=1);

/**
 * CakePHP(tm) : Rapid Development Framework (https://cakephp.org)
 * Copyright (c) Cake Software Foundation, Inc. (https://cakefoundation.org)
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @copyright Copyright (c) Cake Software Foundation, Inc. (https://cakefoundation.org)
 * @link      https://cakephp.org CakePHP(tm) Project
 * @since     0.2.9
 * @license   https://opensource.org/licenses/mit-license.php MIT License
 */
namespace App\Controller;

use Cake\Controller\Controller;
use App\Service\FeatureService;
use Cake\Core\Configure;

/**
 * Application Controller
 *
 * Add your application-wide methods in the class below, your controllers
 * will inherit them.
 *
 * @link https://book.cakephp.org/5/en/controllers.html#the-app-controller
 */
class AppController extends Controller
{
    /**
     * Initialization hook method.
     *
     * Use this method to add common initialization code like loading components.
     *
     * e.g. `$this->loadComponent('FormProtection');`
     *
     * @return void
     */
    public function initialize(): void
    {
        parent::initialize();

        $this->loadComponent('Flash');
        $this->loadComponent('Authentication.Authentication');

        /*
         * Enable the following component for recommended CakePHP form protection settings.
         * see https://book.cakephp.org/5/en/controllers/components/form-protection.html
         */
        //$this->loadComponent('FormProtection');
    }

    public function beforeRender(\Cake\Event\EventInterface $event)
    {
        parent::beforeRender($event);

        Configure::write('CurrentClinicId', $this->currentClinicId());

        $identity = $this->request->getAttribute('identity');
        $role = strtolower((string)($identity ? $identity->get('role') : 'clinician'));

        $role = $this->currentRole();
        $fs   = new FeatureService();
        $featureVis = $fs->visibleMapForRole($role);

        $this->set('featureVisibility', $featureVis);
        $this->set('isAdmin', $role === 'admin');
        $this->set('featureVis', $fs->visibleMapForRole($role));
    }

    protected function currentRole(): string
    {
        $identity = $this->request->getAttribute('identity');
        if ($identity && method_exists($identity, 'get')) {
            $val = (string)$identity->get('role');
            if ($val !== '') return strtolower($val);
        }
        return 'clinician';
    }

    public function currentClinicId(): ?int
    {
        $session = $this->request->getSession();
        $existing = $session->read('Current.clinic_id');
        if ($existing !== null) {
            return (int)$existing;
        }

        $identity = $this->request->getAttribute('identity');
        if (!$identity) {
            return null;
        }

        // Find a clinic linked to this login via clinic_users
        $ClinicUsers = $this->fetchTable('ClinicUsers');
        $link = $ClinicUsers->find()
            ->select(['clinic_id'])
            ->where(['login_id' => (int)$identity->getIdentifier()])
            ->first();

        if ($link) {
            $session->write('Current.clinic_id', (int)$link->clinic_id);
            return (int)$link->clinic_id;
        }

        return null;
    }

}
