<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\Http\Response;

class OnboardingController extends AppController
{
    public function initialize(): void
    {
        parent::initialize();
        $this->OnboardingStates = $this->fetchTable('OnboardingStates');
        $this->viewBuilder()->setLayout('default');
    }

    /** Resolve clinic id from session or the first clinic for the current user. */
    private function resolveClinicId(): ?int
    {
        $sess = $this->request->getSession();
        $cid  = $sess->read('Current.clinic_id');
        if ($cid) return (int)$cid;

        $identity = $this->request->getAttribute('identity');
        if ($identity) {
            $loginId = (int)$identity->getIdentifier();
            $ClinicUsers = $this->fetchTable('ClinicUsers'); // maps to clinic_users
            $row = $ClinicUsers->find()
                ->select(['clinic_id'])
                ->where(['login_id' => $loginId])
                ->orderDesc('role')
                ->first();

            if ($row && $row->clinic_id) {
                $cid = (int)$row->clinic_id;
                $sess->write('Current.clinic_id', $cid);
                return $cid;
            }
        }
        return null;
    }

    /** Get or create the onboarding state row for the clinic. */
    private function getOrCreateState(int $clinicId)
    {
        $state = $this->OnboardingStates->find()
            ->where(['clinic_id' => $clinicId])
            ->first();

        if (!$state) {
            $state = $this->OnboardingStates->newEntity([
                'clinic_id'  => $clinicId,
                'dismissed'  => 0,
                'step1_done' => 0,
                'step2_done' => 0,
                'step3_done' => 0,
                'step4_done' => 0,
            ]);
            $this->OnboardingStates->save($state);
        }
        return $state;
    }

    public function index()
    {
        $cid = $this->resolveClinicId();
        if (!$cid) {
            return $this->redirect(['controller' => 'Calendar', 'action' => 'index']);
        }

        $state = $this->getOrCreateState($cid);
        $this->set(compact('state'));
    }

    /** NEW: AJAX body for the modal */
    public function modal()
    {
        $this->request->allowMethod(['get']);

        $cid = $this->resolveClinicId();
        //Log::debug('completeStep: cid=' . var_export($cid,true) . ' step=' . var_export($step,true));

        if (!$cid) {
            return $this->response->withStatus(400)->withStringBody('No clinic selected');
        }

        $state = $this->getOrCreateState($cid);
        $this->set(compact('state'));

        // Render the modal partial without the site layout
        $this->viewBuilder()->setLayout('ajax'); // or ->disableAutoLayout()
        return $this->render('modal');           // templates/Onboarding/modal.php
    }

    /** Mark a step done (called from the modal). */
    public function completeStep(?int $step = null)
    {
        $this->request->allowMethod(['get','post','patch','put']);

        if (!$step || $step < 1 || $step > 4) {
            return $this->response->withStatus(400)->withType('application/json')
                ->withStringBody(json_encode(['ok' => false, 'error' => 'Invalid step']));
        }

        $cid = $this->resolveClinicId();
        if (!$cid) {
            return $this->response->withStatus(400)->withType('application/json')
                ->withStringBody(json_encode(['ok' => false, 'error' => 'No clinic selected']));
        }

        $state = $this->getOrCreateState($cid);
        $field = 'step' . $step . '_done';
        if (!(int)$state->get($field)) {
            $state->set($field, 1);
            $this->OnboardingStates->save($state);
        }

        $done    = (int)$state->step1_done + (int)$state->step2_done + (int)$state->step3_done + (int)$state->step4_done;
        $percent = (int)round($done * 25);

        return $this->response->withType('application/json')
            ->withStringBody(json_encode(['ok' => true, 'step' => $step, 'percent' => $percent]));
    }

    public function dismiss()
    {
        $cid = $this->resolveClinicId();
        if (!$cid) return $this->redirect(['controller' => 'Calendar', 'action' => 'index']);

        $state = $this->getOrCreateState($cid);
        $state->dismissed = 1;
        $this->OnboardingStates->save($state);

        return $this->redirect(['controller' => 'Calendar', 'action' => 'index']);
    }

    public function undismiss()
    {
        $cid = $this->resolveClinicId();
        if ($cid) {
            $state = $this->getOrCreateState($cid);
            $state->dismissed = 0;
            $this->OnboardingStates->save($state);
            $this->Flash->success('Onboarding banner will appear again.');
        }
        return $this->redirect(['controller' => 'Calendar', 'action' => 'index']);
    }
}
