<?php
declare(strict_types=1);

namespace App\Controller;

use Authentication\Controller\Component\AuthenticationComponent;
use Authentication\PasswordHasher\DefaultPasswordHasher;
use Cake\I18n\FrozenTime;
use Cake\Mailer\Mailer;
use Cake\Routing\Router;
use Cake\Utility\Security;

class LoginController extends AppController
{
    /** @var AuthenticationComponent */
    public $Authentication;

    public function initialize(): void
    {
        parent::initialize();

        /** @var \Authentication\Controller\Component\AuthenticationComponent $auth */
        $auth = $this->loadComponent('Authentication.Authentication');
        $this->Authentication = $auth;

        if (method_exists($auth, 'addUnauthenticatedActions')) {
            $auth->addUnauthenticatedActions(['index','register', 'registerClinic','forgot','reset','logout']);
        } else {
            $auth->allowUnauthenticated(['index','register', 'registerClinic','forgot','reset','logout']);
        }



        $this->viewBuilder()->setLayout('Login');
        $this->viewBuilder()->disableAutoLayout();
    }

    public function index()
    {
        $this->request->allowMethod(['get', 'post']);

        if ($this->request->is('post')) {
            $recaptchaResponse = $this->request->getData('g-recaptcha-response') ?? '';
            $secretKey = '6Le6pqQrAAAAAJPa5lcyrNNPgMCdszRtviK3XhOE';

            $verifyUrl = "https://www.google.com/recaptcha/api/siteverify?secret={$secretKey}&response={$recaptchaResponse}";
            $verifyResponse = file_get_contents($verifyUrl);
            $responseData = json_decode($verifyResponse);

            if (empty($responseData) || !$responseData->success) {
                $this->Flash->error('reCAPTCHA verification failed. Please try again.');
                return $this->redirect(['action' => 'index']);
            }

            $result = $this->Authentication->getResult();

            if ($result && $result->isValid()) {
                // refresh current clinic context
                $this->request->getSession()->delete('Current.clinic_id');
                $this->currentClinicId();

                $target = $this->request->getQuery('redirect') ?: ['controller' => 'Calendar', 'action' => 'index'];
                if ($target === '/' || $target === ['controller' => 'Login', 'action' => 'index']) {
                    $target = ['controller' => 'Calendar', 'action' => 'index'];
                }
                return $this->redirect($target);
            }

            // failed login
            $this->Flash->error('Invalid email or password.');
            return $this->redirect(['action' => 'index']);
        }
    }

    public function register()
    {
        return $this->redirect(['controller' => 'ClinicianRegister', 'action' => 'index']);
    }

    public function registerClinic()
    {
        $this->request->allowMethod(['post']);

        $Login            = $this->fetchTable('Login');
        $Clinics          = $this->fetchTable('Clinics');
        $ClinicUsers      = $this->fetchTable('ClinicUsers');
        $OnboardingStates = $this->fetchTable('OnboardingStates');

        $raw = $this->request->getData();

        $firstName = trim((string)($raw['first_name'] ?? ''));
        $lastName  = trim((string)($raw['last_name'] ?? ''));
        $email     = trim((string)($raw['email'] ?? ''));
        $password  = (string)($raw['password'] ?? '');

        if ($firstName === '' || $lastName === '') {
            $this->Flash->error('First and last name are required.');
            return $this->redirect(['action' => 'index']);
        }
        if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->Flash->error('Please enter a valid email address.');
            return $this->redirect(['action' => 'index']);
        }

        if ($Login->exists(['email' => $email])) {
            $this->Flash->error('An account with that email already exists.');
            return $this->redirect(['action' => 'index']);
        }

        if (!(preg_match('/[A-Z]/', $password) && preg_match('/\d/', $password) && strlen($password) >= 8)) {
            $this->Flash->error('Password must be at least 8 characters and contain at least 1 uppercase letter and 1 number.');
            return $this->redirect(['action' => 'index']);
        }

        $clinicName     = trim((string)($raw['clinic_name'] ?? '')) ?: ($firstName ? "{$firstName}'s Clinic" : 'New Clinic');
        $clinicTz       = trim((string)($raw['timezone'] ?? 'Australia/Sydney'));
        $clinicBizPhone = trim((string)($raw['business_phone'] ?? ''));

        $admin = $Login->patchEntity($Login->newEmptyEntity(), [
            'first_name'    => $firstName,
            'last_name'     => $lastName,
            'email'         => $email,
            'password'      => $password,
            'role'          => 'admin',
            'mobile_phone'  => '000000000',
            'date_of_birth' => date('Y-m-d'),
            'terms_agreed'  => 1,
        ]);

        if ($admin->getErrors()) {
            $this->Flash->error('The input information is incorrect. Please try again.');
            return $this->redirect(['action' => 'index']);
        }

        $conn = $Login->getConnection();
        $conn->begin();

        try {
            $Login->saveOrFail($admin);

            $clinic = $Clinics->patchEntity($Clinics->newEmptyEntity(), [
                'name'           => $clinicName,
                'timezone'       => $clinicTz,
                'business_phone' => $clinicBizPhone ?: null,
            ]);
            $Clinics->saveOrFail($clinic);

            $ownerLink = $ClinicUsers->patchEntity($ClinicUsers->newEmptyEntity(), [
                'clinic_id' => $clinic->id,
                'login_id'  => $admin->id,
                'role'      => 'owner',
            ]);
            $ClinicUsers->saveOrFail($ownerLink);

            $obs = $OnboardingStates->patchEntity($OnboardingStates->newEmptyEntity(), [
                'clinic_id'      => $clinic->id,
                'owner_login_id' => $admin->id,
            ]);
            $OnboardingStates->saveOrFail($obs);

            $conn->commit();

            $this->Flash->success('Admin account and clinic created successfully. You can now log in.');
            return $this->redirect(['action' => 'index']);
        } catch (\Throwable $e) {
            $conn->rollback();

            $this->Flash->error('Failed to create the account and clinic. Please try again.');
            return $this->redirect(['action' => 'index']);
        }
    }

    public function logout()
    {
        $this->request->allowMethod(['get','post']);
        $this->Authentication->logout();
        $this->Flash->success('You have been logged out.');
        return $this->redirect('/');
    }

    private function makeTokenPair(): array
    {
        $plain = bin2hex(random_bytes(32));
        $hmac  = hash_hmac('sha256', $plain, Security::getSalt());
        return [$plain, $hmac];
    }

    private function hmac(string $plain): string
    {
        return hash_hmac('sha256', $plain, Security::getSalt());
    }

    public function forgot()
    {
        $this->request->allowMethod(['get', 'post']);
        $users = $this->fetchTable('Login');

        if ($this->request->is('post')) {
            $email = trim((string)$this->request->getData('email'));
            if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $this->Flash->error('Please enter a valid email address.');
                return;
            }

            $flashOk = function () {
                $this->Flash->success('If that email exists, a reset link has been sent.');
                return $this->redirect(['action' => 'index']);
            };

            $user = $users->find()->where(['email' => $email])->first();
            if (!$user) {
                return $flashOk();
            }

            [$plainToken, $hmac] = $this->makeTokenPair();
            $user->reset_token = $hmac;
            $user->reset_token_expires = FrozenTime::now()->addHours(2);

            if ($users->save($user)) {
                $link = Router::url(['controller' => 'Login', 'action' => 'reset', $plainToken], true);
                try {
                    (new Mailer('default'))
                        ->setTo($user->email)
                        ->setSubject('Password Reset')
                        ->deliver("Click this link to reset your password:\n{$link}\nThis link expires in 2 hours.");
                } catch (\Throwable $e) {
                }
            }

            return $flashOk();
        }
    }

    public function reset(string $token)
    {
        $this->request->allowMethod(['get', 'post']);
        $users = $this->fetchTable('Login');

        $hmac = $this->hmac($token);
        $user = $users->find()
            ->where([
                'reset_token' => $hmac,
                'reset_token_expires >' => FrozenTime::now(),
            ])->first();

        if (!$user) {
            $this->Flash->error('The reset link is invalid or has expired.');
            return $this->redirect(['action' => 'forgot']);
        }

        if ($this->request->is('post')) {
            $pw  = (string)$this->request->getData('password');
            $pw2 = (string)$this->request->getData('password_confirm');

            if ($pw === '' || $pw !== $pw2) {
                $this->Flash->error('Passwords do not match.');
                return;
            }
            if (strlen($pw) < 8) {
                $this->Flash->error('Password must be at least 8 characters.');
                return;
            }

            $hasher = new DefaultPasswordHasher();
            $user->password = $hasher->hash($pw);
            $user->reset_token = null;
            $user->reset_token_expires = null;

            if ($users->save($user)) {
                $this->Flash->success('Password has been reset. You can now log in.');
                return $this->redirect(['action' => 'index']);
            }

            $this->Flash->error('Failed to reset password. Please try again.');
        }

        $this->set(compact('token'));
    }

    public function new()
    {

    }
}

