<?php
declare(strict_types=1);

namespace App\Controller;

use Authentication\PasswordHasher\DefaultPasswordHasher;
use Cake\Http\Response;
use Cake\I18n\FrozenTime;

class ClinicianController extends AppController
{
    /** @var \Cake\ORM\Table */
    private \Cake\ORM\Table $CalendarEvents;
    /** @var \Cake\ORM\Table */
    private \Cake\ORM\Table $Login;
    /** @var \Cake\ORM\Table */
    private \Cake\ORM\Table $ClinicUsers;

    private function isAdmin(): bool
    {
        $id = $this->request->getAttribute('identity');

        return $id && strtolower((string)($id->get('role') ?? '')) === 'admin';
    }

    private function meId(): ?int
    {
        $id = $this->request->getAttribute('identity');

        return $id ? (int)$id->getIdentifier() : null;
    }

    public function initialize(): void
    {
        parent::initialize();
        $this->loadComponent('Authentication.Authentication');

        $this->CalendarEvents = $this->fetchTable('CalendarEvents');
        $this->Login          = $this->fetchTable('Login');
        $this->ClinicUsers    = $this->fetchTable('ClinicUsers');

        $this->viewBuilder()->setLayout('default');
    }

    public function beforeFilter(\Cake\Event\EventInterface $event)
    {
        parent::beforeFilter($event);

        if ($this->components()->has('Authentication')) {
            $result = $this->Authentication->getResult();
            if (!$result || !$result->isValid()) {
                return $this->redirect([
                    'controller' => 'Login',
                    'action'     => 'index',
                    '?'          => ['redirect' => $this->request->getRequestTarget()],
                ]);
            }
        }
    }

    /**
     * List clinicians that belong to the current clinic (via clinic_users).
     */
    public function index()
    {
        $this->viewBuilder()->setLayout('default');

        $q       = trim((string)$this->request->getQuery('q', ''));
        $perPage = (int)$this->request->getQuery('limit', 20);
        if ($perPage <= 0) { $perPage = 20; }

        $clinicId = $this->currentClinicId();

        $query = $this->Login->find()
            ->select([
                'Login.id', 'Login.first_name', 'Login.last_name',
                'Login.email', 'Login.mobile_phone', 'Login.created'
            ])
            ->where(['Login.role' => 'clinician'])
            // scope to clinic via clinic_users
            ->innerJoin(['cu' => 'clinic_users'], 'cu.login_id = Login.id')
            ->where(['cu.clinic_id' => $clinicId])
            ->contain(['Specializations']);

        if ($q !== '') {
            $query
                ->leftJoinWith('Specializations')
                ->where([
                    'OR' => [
                        'Login.first_name LIKE'     => "%{$q}%",
                        'Login.last_name LIKE'      => "%{$q}%",
                        'Login.email LIKE'          => "%{$q}%",
                        'Login.mobile_phone LIKE'   => "%{$q}%",
                        'Specializations.name LIKE' => "%{$q}%",
                    ],
                ])
                ->group(['Login.id']);
        }

        $this->paginate = ['limit' => $perPage, 'order' => ['Login.id' => 'ASC']];
        $clinicians = $this->paginate($query);

        $this->set(compact('clinicians', 'q', 'perPage'));
    }

    private function requireAdmin(): ?Response
    {
        $result   = $this->Authentication->getResult();
        $identity = $this->request->getAttribute('identity');

        if (!$result || !$result->isValid() || !$identity) {
            $this->Flash->error('Please login first.');
            return $this->redirect(['controller' => 'Login', 'action' => 'index']);
        }

        if (strtolower((string)$identity->get('role')) !== 'admin') {
            $this->Flash->error('Only administrators can perform this action.');
            return $this->redirect(['action' => 'index']);
        }

        return null;
    }

    /**
     * Ensure the clinician id belongs to the current clinic (for admin-on-others).
     */
    private function assertInCurrentClinic(int $loginId): void
    {
        $clinicId = $this->currentClinicId();

        $link = $this->ClinicUsers->find()
            ->where(['clinic_id' => $clinicId, 'login_id' => $loginId])
            ->first();

        if (!$link) {
            // 404 rather than leaking that the record exists in another clinic
            throw new \Cake\Http\Exception\NotFoundException('Clinician not found.');
        }
    }

    public function edit(int $id)
    {
        $Logins          = $this->Login;
        $Specializations = $this->fetchTable('Specializations');

        $Admin = $this->isAdmin();
        if (!$Admin && $this->meId() !== $id) {
            $this->Flash->error('You can only edit your own profile.');
            return $this->redirect(['action' => 'profile']);
        }

        // If admin is editing someone else, verify they're in the current clinic
        if ($Admin && $this->meId() !== $id) {
            $this->assertInCurrentClinic($id);
        }

        // Build query
        $query = $Logins->find()
            ->where(['id' => $id])
            ->contain(['Specializations']);

        // If admin is editing *someone else*, enforce role=clinician
        if ($Admin && $this->meId() !== $id) {
            $query->where(['role' => 'clinician']);
        }

        $clinician = $query->firstOrFail();

        $identity = $this->request->getAttribute('identity');
        $isAdmin  = strtolower((string)($identity->get('role') ?? '')) === 'admin';

        $backUrl = $isAdmin ? ['action' => 'index'] : ['controller' => 'Clinician', 'action' => 'profile'];

        if ($this->request->getQuery('return') === 'profile') {
            $profileId = (int)($this->request->getQuery('id') ?: $id);
            $backUrl   = ['controller' => 'Clinician', 'action' => 'profile', '?' => ['id' => $profileId]];
        }

        $ref = (string)($this->request->getQuery('back') ?: $this->request->referer());
        if (strpos($ref, '/clinicians') !== false) {
            $backUrl = ['controller' => 'Clinician', 'action' => 'index'];
        }

        $specializations = $Specializations->find('list', [
            'keyField'   => 'id',
            'valueField' => 'name',
        ])->toArray();

        if ($this->request->is(['patch', 'post', 'put'])) {
            if ($this->request->getData('cancel')) {
                return $this->redirect($backUrl);
            }

            $data = $this->request->getData();

            // Only force role=clinician if admin is editing another clinician
            if ($Admin && $this->meId() !== $id) {
                $data['role'] = 'clinician';
            }

            if (isset($data['specializations_ids'])) {
                $data['specializations'] = [
                    '_ids' => array_map('intval', (array)$data['specializations_ids']),
                ];
                unset($data['specializations_ids']);
            }

            $clinician = $Logins->patchEntity($clinician, $data, [
                'associated' => ['Specializations'],
            ]);

            if ($Logins->save($clinician)) {
                $this->Flash->success('Clinician updated.');
                return $this->redirect($backUrl);
            }
            $this->Flash->error('Could not update clinician. Check the form for errors.');
        }

        $this->set(compact('clinician', 'specializations', 'backUrl'));
        $this->viewBuilder()->setLayout('default');
    }

    public function delete(int $id)
    {
        if ($resp = $this->requireAdmin()) { return $resp; }
        $this->request->allowMethod(['post', 'delete']);

        // Ensure the clinician is in the current clinic
        $this->assertInCurrentClinic($id);

        $clinician = $this->Login->find()
            ->where(['id' => $id, 'role' => 'clinician'])
            ->firstOrFail();

        if ($this->Login->delete($clinician)) {
            $this->Flash->success('Clinician deleted.');
        } else {
            $this->Flash->error('Clinician could not be deleted.');
        }

        return $this->redirect(['action' => 'index']);
    }

    public function profile(?int $id = null)
    {
        $Logins    = $this->Login;
        $identity  = $this->request->getAttribute('identity');
        $viewerId  = (int)$identity->getIdentifier();
        $isAdmin   = strtolower((string)($identity->get('role') ?? '')) === 'admin';

        // prefer route param $id, then ?id=..., else current user
        $targetId = $id ?? (int)($this->request->getQuery('id') ?: 0) ?: $viewerId;

        // Non-admins cannot view others
        if (!$isAdmin && $targetId !== $viewerId) {
            $targetId = $viewerId;
        }

        // If admin is viewing *another* user, they must be a clinician in this clinic
        if ($isAdmin && $targetId !== $viewerId) {
            $this->assertInCurrentClinic($targetId);
        }

        // Load the user
        $query = $Logins->find()->where(['id' => $targetId]);
        if ($isAdmin && $targetId !== $viewerId) {
            $query->where(['role' => 'clinician']);
        }
        $user = $query->firstOrFail();

        $this->set([
            'user'         => $user,
            'firstName'    => (string)$user->first_name,
            'lastName'     => (string)$user->last_name,
            'email'        => (string)$user->email,
            'mobile'       => (string)$user->mobile_phone,
            'isAdmin'      => $isAdmin,
            'isOwnProfile' => ($targetId === $viewerId),
        ]);

        // Allow updates only on your own profile
        if ($this->request->is(['patch', 'post', 'put']) && $targetId === $viewerId) {
            $user = $Logins->patchEntity($user, $this->request->getData());

            if ($Logins->save($user)) {
                $this->Authentication->setIdentity($user); // refresh session identity
                $this->Flash->success('Profile updated.');
                return $this->redirect(['action' => 'profile']);
            }
            $this->Flash->error('Could not update profile.');
        }

        $this->viewBuilder()->setLayout('default');
        $this->set('title', 'Clinician Profile');
    }

    public function changePassword()
    {
        $this->request->allowMethod(['get', 'post']);

        $Logins   = $this->Login;
        $identity = $this->request->getAttribute('identity');
        if (!$identity) {
            $this->Flash->error('Please sign in first.');
            return $this->redirect(['controller' => 'Login', 'action' => 'index']);
        }

        // Only allow changing *own* password
        $userId = (int)$identity->getIdentifier();
        $user   = $Logins->get($userId);

        if ($this->request->is('post')) {
            $old     = (string)$this->request->getData('old_password');
            $new     = (string)$this->request->getData('new_password');
            $confirm = (string)$this->request->getData('new_password_confirm');

            // 1) Check old password
            $hasher = new DefaultPasswordHasher();
            if (!$hasher->check($old, (string)$user->password)) {
                $this->Flash->error('Your current password is incorrect.');
                return;
            }

            // 2) Checks
            $errors = [];
            if (strlen($new) < 8) {
                $errors[] = 'New password must be at least 8 characters.';
            }
            if (!preg_match('/[A-Z]/', $new)) {
                $errors[] = 'New password must contain at least one uppercase letter.';
            }
            if (!preg_match('/\d/', $new)) {
                $errors[] = 'New password must contain at least one number.';
            }
            if ($new !== $confirm) {
                $errors[] = 'New password and confirmation do not match.';
            }
            if ($old === $new) {
                $errors[] = 'New password must be different from your current password.';
            }

            if ($errors) {
                $this->Flash->error(implode(' ', $errors));
                return;
            }

            // 3) Save
            $user->password = $hasher->hash($new);

            if ($Logins->save($user)) {
                // Refresh session identity so user stays logged in
                $this->Authentication->setIdentity($user);

                $this->Flash->success('Password updated successfully.');
                return $this->redirect(['action' => 'profile']);
            }

            $this->Flash->error('Could not update your password. Please try again.');
        }

        $this->set(compact('user'));
        $this->viewBuilder()->setLayout('default');
    }
}
